/* $Id: scope.c,v 1.4 1998/12/04 02:39:48 ericb Exp $ */
/* Copyright (C) 1997 - 1998 Hewlett-Packard Company, all rights reserved. */

/* Scope-like demo program for E1432/E1433 using PlugNPlay on HP-UX 10

   This program runs a simple auto-arm, auto-trigger time-domain
   measurement.  This makes a nice "warm-fuzzy" test that the E143x
   software is installed properly and your E143x hardware is working.

   This program is nearly identical to the "semascope" program.  The
   difference is that this program also finds source channels, and
   turns them all on.  The source mode defaults to burst random, but
   can be changed on the command-line.

   By default, the program will search the VXI system for all
   available E143x modules.  If necessary, it will download the
   "sema.bin" firmware into the modules, and will then run a simple
   measurement.  The data from the measurement will get plotted to X11
   windows, in real time.

   Use "./semascope -u" for info on command-line options.  The default
   is to use all modules, a blocksize of 1024, an input range of 1
   Volt, a span of 20 kHz, block mode, and plot data for all channels.
   The sources are set to burst random mode, with a 50% duty cycle.
   These defaults can all be changed on the command-line. */

#define CHK \
    if(vierr)\
    {\
        hpe1432_error_message(session,vierr,st);\
        hpe1432_errorDetails(session,details,100);\
        (void) printf("error %d = %s, line %d\nDetails: %s\n",vierr,st,__LINE__,details);\
        exit(0);\
    }

#include <stdio.h>		/* For printf */
#include <stdlib.h>		/* For exit */
#include <string.h>		/* For strrchr */
#include <unistd.h>		/* For getopt */
#include <X11/Xlib.h>		/* For XOpenDisplay */

#include "xplot.h"		/* For xplot_init_plot */
#include "hpe1432.h"		/* For hpe1432_xxx */

/* Hard-coded maximums */
#define	NMODLIST_MAX	64	/* Max E143x modules */
#define	NCHAN_MAX	(NMODLIST_MAX * 16)  /*FIX*/

/* Pixel spacing between windows */
#define	X_SPACING	15
#define	Y_SPACING	40

/* Default values for several parameters */
#define	BLOCKSIZE_DEF	1024
#define	CLOCK_FREQ_DEF	51200.0
#define	DUTY_CYCLE_DEF	0.5
#define	RANGE_DEF	1.0
#define	SPAN_DEF	20000.0


ViStatus	vierr;
ViSession	session;
char		st[100];
char		details[100];

static char rcsid[] =
"@(#)$Id: scope.c,v 1.4 1998/12/04 02:39:48 ericb Exp $";
static char *progname;

/* Initialize E143x library, decide which modules to use, download
   code if needed, create channel group. */
static int
init(ViInt32 nladdr, ViInt32 laddr[], ViInt32 *nmodlist, ViInt32 modlist[],
     ViInt32 nmodlist_user, ViInt32 *nchan, ViInt32 *nsrcchan,
     ViInt32 chanlist[], ViInt32 *group, ViInt32 *srcgroup,
     ViInt32 *allgroup, ViInt32 flipflag)
{
    ViInt32 hwinfo[27*NMODLIST_MAX];
    ViInt32 modlist_tmp[NMODLIST_MAX];
    ViInt32 srcchanlist[NCHAN_MAX];
    ViInt32 allchanlist[2 * NCHAN_MAX];
    int     i, j, chan, nmodlist_tmp;
    ViChar  buf[NMODLIST_MAX * 4 + 32];

    vierr = hpe1432_find(0, modlist, NMODLIST_MAX, nmodlist, buf, 80);
    CHK

    /* --- Decide which modules to use --- */

    if (nladdr != 0)
    {
	/* Use requested logical addresses */
	nmodlist_tmp = 0;
	for (j = 0; j < nladdr; j++)
	    for (i = 0; i < *nmodlist; i++)
		if (modlist[i] == laddr[j])
		    modlist_tmp[nmodlist_tmp++] = laddr[j];
	*nmodlist = nmodlist_tmp;
    }
    else
    {
	/* No requested logical addresses, so use all available */
	for (i = 0; i < *nmodlist; i++)
	    modlist_tmp[i] = modlist[i];
    }

    if (*nmodlist == 0)
    {
	if (nladdr == 0)
	    (void) fprintf(stderr, 
			   "%s: no E143x modules found\n",
			   progname);
	else
	    (void) fprintf(stderr,
			   "%s: no E143x modules match requested LA\n",
			   progname);
	return -1;
    }

    (void) printf("Found %d E143x module%s at:",
		  *nmodlist, *nmodlist > 1 ? "s" : "");
    for (i = 0; i < *nmodlist; i++)
    {
	/* The E1432 library makes the last module in the list of
	   modules the system master.  In order to make the first
	   module the system master, we normally reverse the order of
	   the list.  But if flipflag is specified, then we don't
	   reverse the order. */
	if (flipflag)
	    modlist[i] = modlist_tmp[i];
	else
	    modlist[i] = modlist_tmp[*nmodlist - 1 - i];
	(void) printf(" %d", modlist[i]);
    }
    (void) printf("\n");

    if (nmodlist_user > 0 && nmodlist_user < *nmodlist)
    {
	/* Use number of modules specified by user */
	if (nmodlist_user > 1)
	    (void) printf("First %d modules being used\n", nmodlist_user);
	else
	    (void) printf("First module being used\n");
	*nmodlist = nmodlist_user;
    }

    /* --- Finally done deciding which modules to use --- */

    /* form string to init instrument: "VXI0::mod1,mod2,...::INSTR" */
    (void) strcat(buf, "::");
    for(i = 0; i < *nmodlist; i++)
	(void) sprintf(&buf[strlen(buf)], "%d,", modlist[i]);
    (void) strcpy(&buf[strlen(buf) - 1], "::INSTR"); /*overwrite last "," */
    
    vierr=hpe1432_init(buf, 0, 1, &session);
    CHK

    /* need to find out number of input and source channels */
    vierr = hpe1432_getHWConfig(0, *nmodlist, modlist, hwinfo);
    CHK;

#if 0
    if (vierr != 0)
    {
	(void) printf("Installing sema.bin from "
		      "/opt/e1432/lib/sema.bin in all E143x modules ... ");
	(void) fflush(stdout);
	vierr = hpe1432_install(*nmodlist, modlist, 0,
			    "/opt/e1432/lib/sema.bin");
    	CHK
	(void) printf("done\n");
	vierr = hpe1432_getHWConfig(0, *nmodlist, modlist, hwinfo);
    	CHK
    }
    
    vierr = hpe1432_assignChannelNumbers(*nmodlist, modlist, session);
    CHK

#endif

    *nchan = 0;
    *nsrcchan = 0;
    for (i = 0; i < *nmodlist; i++)
    {
	*nchan += hwinfo[27*i+21];
	*nsrcchan += hwinfo[27*i+22];
	switch (hwinfo[27*i+1])
	{
	case HPE1432_MODEL_CODE_E1432:
	    (void) printf("E1432 ");
	    break;
	case HPE1432_MODEL_CODE_E1433:
	    (void) printf("E1433 ");
	    break;
	case HPE1432_MODEL_CODE_E1434:
	    (void) printf("E1434 ");
	    break;
	default:
	    (void) printf("E143x ");
	    break;
	}
	(void) printf("at LA %3d has %2d input channels, %d source channel%s\n",
		      modlist[i], hwinfo[27*i+21], hwinfo[27*i+22],
		      hwinfo[27*i+22] == 1 ? "" : "s");
    }
    if (*nmodlist > 1)
	(void) printf("Total %d input channels\n", *nchan);
    if (*nchan == 0)
    {
	(void) fprintf(stderr, "%s: no input channels found\n",
		       progname);
	return -1;
    }

    for (chan = 0; chan < *nchan; chan++)
	chanlist[chan] = (HPE1432_CHAN_TYPE_INPUT << 12) + chan + 1; 
    for (chan = 0; chan < *nsrcchan; chan++)
	srcchanlist[chan] = (HPE1432_CHAN_TYPE_SOURCE << 12) + chan + 1;
    for (chan = 0; chan < *nchan; chan++)
	allchanlist[chan] = chanlist[chan];
    for (chan = 0; chan < *nsrcchan; chan++)
	allchanlist[*nchan + chan] = srcchanlist[chan];

    vierr = hpe1432_createChannelGroup(session, *nchan, chanlist, group);
    CHK
    if (*group >= 0)
    {
	(void) fprintf(stderr, "%s: hpe1432_createChannelGroup(inputs): "
		       "returned %d\n",
		       progname, *group);
	return -1;
    }

    vierr = hpe1432_createChannelGroup(session, *nsrcchan, 
							srcchanlist, srcgroup);
    CHK
    
    if (*srcgroup >= 0)
    {
	(void) fprintf(stderr, "%s: hpe1432_createChannelGroup(sources): "
		       "returned %d\n",
		       progname, *srcgroup);
	return -1;
    }
    
    vierr = hpe1432_createChannelGroup(session, *nchan + *nsrcchan,
					   allchanlist, allgroup);
    CHK
    if (*allgroup >= 0)
    {
	(void) fprintf(stderr, "%s: hpe1432_createChannelGroup(all): "
		       "returned %d\n",
		       progname, *allgroup);
	return -1;
    }

    return 0;
}

static int
setup(ViInt32 group, ViInt32 srcgroup, ViInt32 allgroup, ViInt32 nsrcchan,
      ViInt32 contflag, ViInt32 autozeroflag, ViInt32 combautozeroflag,
      ViPInt32 blocksize, ViReal64 range, ViReal64 clock_freq, ViReal64 *span,
      ViInt32 srcmode, ViInt32 srczoomfactor, ViReal64 duty_cycle)
{
    ViInt32 tmp_blocksize;
    ViReal64 tmp_clock_freq, tmp_range, tmp_span;

    /* Input and Source setups */
    vierr = hpe1432_setArmMode(session, allgroup, HPE1432_MANUAL_ARM);
    CHK
    vierr = hpe1432_setAutoTrigger(session, allgroup, HPE1432_MANUAL_TRIGGER);
    CHK
    vierr = hpe1432_setClockFreq(session, allgroup, clock_freq);
    CHK
    vierr = hpe1432_setDataFormat(session, allgroup, *blocksize,
				HPE1432_DATA_SIZE_16,
				contflag ? HPE1432_CONTINUOUS_MODE :
				HPE1432_BLOCK_MODE,
				HPE1432_APPEND_STATUS_OFF);
    CHK
    vierr = hpe1432_setSpan(session, allgroup, *span);
    CHK

    /* Input-only setups */
    vierr = hpe1432_setAnalogInput(session, group, HPE1432_INPUT_MODE_VOLT,
				 HPE1432_INPUT_HIGH_NORMAL,
				 HPE1432_ANTI_ALIAS_ANALOG_ON,
				 HPE1432_COUPLING_DC, range);
    CHK
    vierr = hpe1432_setTrigger(session, group, HPE1432_CHANNEL_OFF,
			    0.0, 0.0, 0.0, HPE1432_TRIGGER_SLOPE_POS,
			    HPE1432_TRIGGER_MODE_LEVEL);
    CHK

    /* Source-only setups */
    vierr = hpe1432_setActive(session, srcgroup, HPE1432_CHANNEL_ON);
    CHK
    vierr = hpe1432_setDutyCycle(session, srcgroup, duty_cycle);
    CHK
    vierr = hpe1432_setRampRate(session, srcgroup, 0);
    CHK
    vierr = hpe1432_setRange(session, srcgroup, 1);
    CHK
    vierr = hpe1432_setSineFreq(session, srcgroup, *span * 2.56 / 64);
    CHK
    vierr = hpe1432_setSourceBlocksize(session, srcgroup,
				     *blocksize / srczoomfactor);
    CHK
    vierr = hpe1432_setSourceCenterFreq(session, srcgroup, 2000);
    CHK
    vierr = hpe1432_setSourceMode(session, srcgroup, srcmode);
    CHK
    vierr = hpe1432_setSourceSpan(session, srcgroup, *span / srczoomfactor);
    CHK
    if (nsrcchan > 0 &&
	(srcmode == HPE1432_SOURCE_MODE_BRAND ||
	 srcmode == HPE1432_SOURCE_MODE_BRANDZ ||
	 srcmode == HPE1432_SOURCE_MODE_BSINE))
    {
	/* Burst source, let the source trigger the measurement.  Use
	   last source, because in multi-mainframe systems this is
	   more likely to be in the root mainframe.  If it is not in
	   the root mainframe, it won't work properly. */
	vierr = hpe1432_setTriggerChannel(session, 
				(HPE1432_CHAN_TYPE_SOURCE << 12) + nsrcchan,
				HPE1432_CHANNEL_ON);
        CHK
    }

    /* Print setup */
    vierr = hpe1432_getBlocksize(session, 1, &tmp_blocksize);
    CHK
    vierr = hpe1432_getClockFreq(session, 1, &tmp_clock_freq);
    CHK
    vierr = hpe1432_getRange(session, 1, &tmp_range);
    CHK
    vierr = hpe1432_getSpan(session, 1, &tmp_span);
    CHK
    *blocksize = tmp_blocksize;
    *span = tmp_span;
    if (nsrcchan > 0 &&
	(srcmode == HPE1432_SOURCE_MODE_BRAND ||
	 srcmode == HPE1432_SOURCE_MODE_BRANDZ ||
	 srcmode == HPE1432_SOURCE_MODE_BSINE))
	(void) printf("Trigger:\tSource\n");
    else
	(void) printf("Trigger:\tManual\n");
    (void) printf("Blocksize:\t%ld\n", tmp_blocksize);
    (void) printf("Range:\t\t%g Volts\n", tmp_range);
    (void) printf("Clock Freq:\t%g Hz\n", tmp_clock_freq);
    (void) printf("Span:\t\t%g Hz\n", tmp_span);
    (void) printf("Data Mode:\t%s\n",
		  contflag ? "Continuous" : "Block");
    switch (srcmode)
    {
    case HPE1432_SOURCE_MODE_RAND:
	(void) printf("Src Mode:\tRandom\n");
	break;
    case HPE1432_SOURCE_MODE_BRAND:
	(void) printf("Src Mode:\tBurst Random\n");
	(void) printf("Duty Cycle:\t%g\n", duty_cycle);
	break;
    case HPE1432_SOURCE_MODE_RANDZ:
	(void) printf("Src Mode:\tZoom Random\n");
	break;
    case HPE1432_SOURCE_MODE_BRANDZ:
	(void) printf("Src Mode:\tBurst Zoom Random\n");
	(void) printf("Duty Cycle:\t%g\n", duty_cycle);
	break;
    case HPE1432_SOURCE_MODE_SINE:
	(void) printf("Src Mode:\tSine\n");
	break;
    case HPE1432_SOURCE_MODE_BSINE:
	(void) printf("Src Mode:\tBurst Sine\n");
	(void) printf("Duty Cycle:\t%g\n", duty_cycle);
	break;
    }

    if (autozeroflag)
    {
	(void) printf("Doing input autozero ... ");
	(void) fflush(stdout);
	vierr = hpe1432_autoZero(session, group);
        CHK
	(void) printf("done\n");

	(void) printf("Doing source autozero ... ");
	(void) fflush(stdout);
	vierr = hpe1432_autoZero(session, srcgroup);
        CHK
	(void) printf("done\n");
    }
    if (combautozeroflag)
    {
	(void) printf("Doing input and source autozero ... ");
	(void) fflush(stdout);
	vierr = hpe1432_autoZero(session, allgroup);
        CHK
	(void) printf("done\n");
    }

    return 0;
}

static int
allocate(ViInt32 blocksize, ViInt32 nchan, ViReal32 **dataBuffer)
{
    /* Use calloc, so that the buffer holds zeros before we get real
       data, to keep the xplot code happy before the first data
       block. */
    *dataBuffer = calloc(sizeof **dataBuffer, blocksize * nchan);
    if (*dataBuffer == NULL)
    {
	(void) fprintf(stderr,
		       "%s: unable to allocate buffer",
		       progname);
	return -1;
    }

    return 0;
}

/* Open plot windows */
static int
open_windows(ViInt32 nchan_plot, ViInt32 chanlist[], char *plotID[],
	     ViReal32 *dataBuffer, ViReal64 span,
	     ViInt32 blocksize, ViReal64 yscale)
{
    Display *disp;
    char    geometry[32], title[8];
    int     divisor, chan;
    int     xscreen, yscreen, xwidth, ywidth, xstart, ystart;

    /* Get screen size in pixels */
    xscreen = 0;
    yscreen = 0;
    disp = XOpenDisplay("");
    if (disp != 0)
    {
	xscreen = XDisplayWidth(disp, 0);
	yscreen = XDisplayHeight(disp, 0);
	(void) XCloseDisplay(disp);
    }
    if (xscreen == 0 || yscreen == 0)
    {
	/* If anything failed, guess at a size */
	xscreen = 1024;
	yscreen = 768;
    }

    divisor = 2;	/* Maximum window size 1/2 screen dimension */
    while ((divisor * divisor) < nchan_plot)
	divisor++;

    /* Window size in pixels */
    xwidth = xscreen / divisor - X_SPACING;
    ywidth = yscreen / divisor - Y_SPACING;

    for (chan = 0; chan < nchan_plot; chan++)
    {
	/* Window location in pixels */
	xstart = chan % divisor * (xwidth + X_SPACING) + 2;
	ystart = chan / divisor * (ywidth + Y_SPACING) + 3;

	(void) sprintf(geometry, "%dx%d+%d+%d",
		       xwidth, ywidth, xstart, ystart);
	(void) sprintf(title, "%3d", chanlist[chan]);

	plotID[chan] = xplot_init_plot(dataBuffer + chan * blocksize,
				       blocksize, span,
				       yscale, -yscale, TIME_TRACE,
				       geometry, title);
    }

    return 0;
}

static int
run(ViInt32 group, ViInt32 allgroup, ViInt32 contflag,
    ViInt32 blocksize, ViInt32 srcmode, ViInt32 nchan, ViInt32 nsrcchan, 
    ViInt32 nchan_plot, ViReal32 *dataBuffer, char *plotID[])
{
    ViInt32 actualsize;
    ViInt32 status, chan;
    ViReal64 *tmpBuffer;
    int i;

    vierr = hpe1432_initMeasure(session, allgroup);
    CHK

    tmpBuffer = malloc(sizeof(ViReal64) * blocksize * nchan);
    if (tmpBuffer == NULL)
    {
	(void) fprintf(stderr,
		       "%s: unable to allocate buffer",
		       progname);
	return -1;
    }

    while (1)
    {
	/* Arm and trigger - first time only for continuous mode,
	   every time for block mode */
	if (contflag < 2)
	{
	    vierr = hpe1432_armMeasure(session, allgroup, 1);
    	    CHK

	    /* If there are source channels, and they are in a burst
	       mode, we have set them up to trigger the measurement.
	       Otherwise we will do it manually.  Note that the source
	       triggering won't necessarily work correctly in a
	       multi-mainframe setup, because the trigger must be
	       generated by the 'master' mainframe.

	       Furthermore, when doing anything other than manual
	       trigger in a multi-mainframe measurement, the arm must
	       be done using hpe1432_armMeasureMasterSetup,
	       hpe1432_armMeasureMasterFinish, and
	       hpe1432_armMeasureSlaveFinish, which this program does
	       not do. */
	    if (nsrcchan == 0 ||
		(srcmode != HPE1432_SOURCE_MODE_BRAND &&
		 srcmode != HPE1432_SOURCE_MODE_BRANDZ &&
		 srcmode != HPE1432_SOURCE_MODE_BSINE))
		vierr = hpe1432_triggerMeasure(session, allgroup, 1);
    	        CHK
	}
	if (contflag == 1)
	    contflag = 2;

	/* Wait for data */
	status = 0;
	while (status == 0)
	{
	    vierr = hpe1432_blockAvailable(session, group, &status);
	    CHK
	    /* While no data, check for X events */
	    for (chan = 0; chan < nchan_plot; chan++)
		xplot_check_events(plotID[chan]);
	}

	/* Read the data */
	vierr = hpe1432_readFloat64Data(session, group, HPE1432_TIME_DATA,
				      tmpBuffer, blocksize * nchan,
				      &actualsize, HPE1432_NO_WAIT_FLAG);
    	CHK
	if (actualsize != blocksize * nchan)
	{
	    (void) fprintf(stderr,
			   "%s: hpe1432_readFloat64Data: "
			   "actual count was %ld, expected %ld\n",
			   progname, actualsize, blocksize * nchan);
	    return -1;
	}

	for(i=0; i < blocksize * nchan; i++)
	    dataBuffer[i] = (float)tmpBuffer[i];

	/* Plot the data */
	for (chan = 0; chan < nchan_plot; chan++)
	{
	    xplot_data_update(plotID[chan]);
	    xplot_check_events(plotID[chan]);
	}
    }
    /*NOTREACHED*/
    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "E1432/E1433 multi-channel time display for X11\n"
		   "     Also starts source outputs\n"
		   "Usage: %s [-cfSuVxzZ] [-b blocksize] [-C clock_freq]\n"
		   "[-d duty_cycle] [-l la] [-n nmodlist]\n"
		   "\t[-N nchan] [-r range] [-s span] [-y yscale]\n"
		   "\t-b: Set blocksize to <blocksize>, default 1024\n"
		   "\t-c: Use continuous mode, default is block mode\n"
		   "\t-C: Set clock_freq to <clock_freq>, default 51200\n"
		   "\t-d: Set duty_cycle to <duty_cycle>, default 0.5\n"
		   "\t-f: Reverse (flip) the LA order\n"
		   "\t    (used for some multi-mainframe systems)\n"
		   "\t-l: Use logical address <la>, multiple -l accumulate\n"
		   "\t-n: Use the first <nmodlist> modules, default all\n"
		   "\t-N: Plot the first <nchan> channels, default 16, -1 means all\n"
		   "\t-r: Set range to <range>, default 1 Volt\n"
		   "\t-s: Set span to <span>, default 20 kHz\n"
		   "\t-S: Use source sine mode instead of random\n"
		   "\t-u: Print this usage message\n"
		   "\t-V: Print version info\n"
		   "\t-x: Use source zoom random mode instead of random\n"
		   "\t-y: Set yscale to <yscale>\n"
		   "\t-z: Do autozero\n"
		   "\t-Z: Do combined input and source autozero\n",
		   progname);
    exit(1);
}

int
main(int argc, char *argv[])
{
    ViInt32	modlist[NMODLIST_MAX];
    ViInt32	chanlist[NCHAN_MAX];
    ViReal32 	*dataBuffer;
    ViReal64	clock_freq, duty_cycle, range, span, yscale;
    char	*plotID[NCHAN_MAX];
    char   	*p;
    ViInt32	blocksize;
    ViInt32     laddr[NMODLIST_MAX];   
    ViInt32	c, contflag, flipflag, autozeroflag, combautozeroflag;
    ViInt32	nladdr, nmodlist, nmodlist_user;
    ViInt32	group, srcgroup, allgroup, nchan, nsrcchan, nchan_plot;
    ViInt32	srcmode, srczoom, srcsine, srczoomfactor;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    blocksize = BLOCKSIZE_DEF;
    clock_freq = CLOCK_FREQ_DEF;
    contflag = 0;
    duty_cycle = DUTY_CYCLE_DEF;
    flipflag = 0;
    nladdr = 0;
    nmodlist_user = 0;
    nchan_plot = 16;	/* Only plot first 16 channels by default */
    range = RANGE_DEF;
    span = SPAN_DEF;
    srcsine = 0;
    srczoom = 0;
    yscale = RANGE_DEF;
    autozeroflag = 0;
    combautozeroflag = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "b:cC:d:fl:n:N:r:s:SuVxy:zZ")) != EOF)
	switch (c)
	{
	case 'b':		/* Blocksize */
	    blocksize = strtol(optarg, &p, 0);
	    if (optarg == p || blocksize <= 0)
	    {
		(void) fprintf(stderr,
			       "%s: invalid blocksize: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'c':		/* Continuous mode */
	    contflag = 1;
	    break;
	case 'C':		/* Clock frequency */
	    if (sscanf(optarg, "%lf", &clock_freq) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid duty_cycle: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'd':		/* Duty cycle */
	    if (sscanf(optarg, "%lf", &duty_cycle) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid duty_cycle: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'f':		/* Reverse la order */
	    flipflag = 1;
	    break;
	case 'l':		/* Logical address */
	    if (nladdr < NMODLIST_MAX)
	    {
		laddr[nladdr] = (ViInt32) strtol(optarg, &p, 0);
		if (optarg == p || laddr[nladdr] <= 0 ||
		    laddr[nladdr] > 255)
		{
		    (void) fprintf(stderr,
				   "%s: invalid logical address: '%s'\n",
				   progname, optarg);
		    usage();
		}
		nladdr++;
	    }
	    else
	    {
		(void) fprintf(stderr,
			       "%s: too many logical addresses requested\n",
			       progname);
		usage();
	    }
	    break;
	case 'n':		/* Module count */
	    nmodlist_user = strtol(optarg, &p, 0);
	    if (optarg == p || nmodlist_user <= 0 ||
		nmodlist_user > NMODLIST_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid module count: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':		/* Channels to plot */
	    nchan_plot = strtol(optarg, &p, 0);
	    if (optarg == p || nchan_plot > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid plot channel count: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'r':		/* Input range */
	    if (sscanf(optarg, "%lf", &range) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid range: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    yscale = range;
	    break;
	case 's':		/* Span */
	    if (sscanf(optarg, "%lf", &span) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid span: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'S':		/* Source sine mode */
	    srcsine = 1;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(0);
	case 'x':		/* Source zoom mode */
	    srczoom = 1;
	    break;
	case 'y':		/* Yscale */
	    if (sscanf(optarg, "%lf", &yscale) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid yscale: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'z':		/* Autozero */
	    autozeroflag = 1;
	    break;
	case 'Z':		/* Combined autozero */
	    combautozeroflag = 1;
	    break;
	case 'u':		/* Usage */
	default:
	    usage();
	}

    /* Deduce source mode and zoom factor from various flags */
    srcmode = contflag ? HPE1432_SOURCE_MODE_RAND :
	HPE1432_SOURCE_MODE_BRAND;
    srczoomfactor = 1;
    if (srczoom)
    {
	srcsine = 0;
	srcmode = contflag ? HPE1432_SOURCE_MODE_RANDZ :
	    HPE1432_SOURCE_MODE_BRANDZ;
	srczoomfactor = 20;
    }
    if (srcsine)
	srcmode = contflag ? HPE1432_SOURCE_MODE_SINE :
	    HPE1432_SOURCE_MODE_BSINE;

    /* Initialize the library */
    if (init(nladdr, laddr, &nmodlist, modlist, nmodlist_user, &nchan,
	     &nsrcchan, chanlist, &group, &srcgroup, &allgroup,
	     flipflag) < 0)
	return 2;

    /* Set up the hardware */
    if (setup(group, srcgroup, allgroup, nsrcchan, contflag,
	      autozeroflag, combautozeroflag, &blocksize, range,
	      clock_freq, &span, srcmode, srczoomfactor,
	      duty_cycle) < 0)
	return 2;

    /* Allocate buffers */
    if (allocate(blocksize, nchan, &dataBuffer) < 0)
	return 2;

    /* Open plot windows */
    if (nchan_plot < 0 || nchan_plot > nchan)
	nchan_plot = nchan;
    if (open_windows(nchan_plot, chanlist, plotID, dataBuffer,
		     span, blocksize, yscale) < 0)
	return 2;

    /* Run the measurement */
    if (run(group, allgroup, contflag, blocksize, srcmode,
	    nchan, nsrcchan, nchan_plot, dataBuffer, plotID) < 0)
	return 2;

    return 0;
}
